#include <time.h>
#include "../common/vector.h"
#include "../common/matrix.h"
#include "../common/misc.h"
#include "simulation.h"
#include "solvers.h"
#include "../render/intersect/sphere.h"
#include "../common/debug.h"
#include "../common/colors.h"
#include "../parse/parser.h"
#include "../common/list.h"
#include "../collision/collision.h"

extern scene_data *main_scene;
extern int running;
extern state *current_state;
extern int current_sim;
extern int sim_type;
extern int shadow_cache;
extern double gravity;

double wire_friction = 0;
double bead_mass = 1;

#define CAM main_scene->camera
#define BEAD main_scene->models[2]

//convert an angle to x,y pos
void parametric_to_euclidean(double angle, vector *v)
{
	v->x = sin(angle);
	v->y = cos(angle);
}


//get the friction on the bead
void calc_bead_force(vector *drag, vector *v)
{
	copy_vector(v, drag);
	multiply_vector(drag, drag, -wire_friction);
}

//get the state derivatives for the bead
void calc_bead_acceleration(state *new_state, state *old_state)
{
	vector force;
	
	calc_bead_force(&force, &old_state->velocity);
	
	//acceleration
	multiply_vector(&new_state->velocity, &force, 1/bead_mass);
	new_state->velocity.x -= sin(old_state->position.x) * gravity;
}


//head function for simulation
void run_bead()
{
	int i, j;
	object *obj;
	
	if(!running)
		return;
	
	//loop over all the beads
	for(i=0; main_scene->models[i] != NULL; i++)
	{
		if(main_scene->models[i]->obj_type == SPHERE)
		{
			//update beads
			if(current_sim == RK4)
				run_rk4((void* (*)(state*, state*))&calc_bead_acceleration, &main_scene->models[i]->st);
			
			else if(current_sim == EULER)
				run_euler((void* (*)(state*, state*))&calc_bead_acceleration, &main_scene->models[i]->st);
			
			else if(current_sim == MIDPOINT)
				run_midpoint((void* (*)(state*, state*))&calc_bead_acceleration, &main_scene->models[i]->st);
			
			parametric_to_euclidean(main_scene->models[i]->st.position.x, &main_scene->models[i]->pos);
			main_scene->models[i]->collided = 0;
		}
	}
	
	
	//collide the beads
	for(i=0; main_scene->models[i] != NULL; i++)
	{
		if(main_scene->models[i]->obj_type == SPHERE)
		{
			check_collisions(main_scene->models[i], i);
		}
	}
}



//add a bead on the wire
void add_bead()
{
	double new_radius;
	vector new_position;
	double new_angle;
	int i;
	int max_tries = 2000;
	
	//create a new position
	new_radius = 0.2;
	new_angle = (double)rand()/RAND_MAX*2*M_PI;
	parametric_to_euclidean(new_angle, &new_position);
	
	//try to place the bead on the wire
	for(i=0; i<max_tries; i++)
	{
		//if new position is no good, make another one
		if(touching_other_sphere(new_radius, &new_position))
		{
			if(i == max_tries - 1)
				exit(1);

			new_angle = (double)rand()/RAND_MAX*2*M_PI;
			parametric_to_euclidean(new_angle, &new_position);
		}
		else
			break;
	}
	
	object *obj = sphere_create();
	
	obj->radius = new_radius;
	copy_vector(&new_position, &obj->pos);
	obj->st.position.x = new_angle;
	multiply_vector(&obj->st.velocity, &obj->st.velocity, 0);

	//tessellate the sphere
	obj->polygon = sphere_polygon(obj, 5);	
	add_pqp_data(obj);
	
	//add bead to scene models
	obj->id = main_scene->num_objects;
	printd(NORMAL, "adding object[%i]\n", obj->id);
	main_scene->models[main_scene->num_objects] = obj;
	main_scene->num_objects++;
	
	color_from_number(&obj->amb, obj->id);
	color_from_number(&obj->diff, obj->id);
}



//removes the last added bead from the scene
void remove_bead()
{
	int i;
	object *obj;
	
	//only remove added beads
	if(main_scene->num_objects < 3)
		return;
	
	obj = main_scene->models[main_scene->num_objects-1];
	printd(NORMAL, "removing object[%i]\n", obj->id);

	delete obj->pqp_data;
	obj->pqp_data = NULL;
	free(obj->polygon->points);
	free(obj->polygon->triangles);
	free(obj);
	
	//don't want the shadow cache to check this object again
	shadow_cache = -1;
	
	main_scene->models[main_scene->num_objects-1] = NULL;
	main_scene->num_objects--;
}


//set the primary bead back to (1,0)
void reset_bead_state()
{
	BEAD->st.position.x = M_PI/2;
	parametric_to_euclidean(BEAD->st.position.x, &BEAD->pos);
	multiply_vector(&BEAD->st.velocity, &BEAD->st.position, 0);
	running = 0;
}

//release the primary bead
void release_bead()
{
	//clear old values
	reset_bead_state();
	running = 1;
}

//setup the bead simulation
void setup_bead()
{
	add_bead();  //the primary bead
	reset_bead_state();
	current_sim = MIDPOINT;
}
